/*************************************************************************************************************
 *                                                                                                           *
 *  Copyright (c) 2015 - 2016, Intel Corporation                                                                    *
 *                                                                                                           *
 *  Redistribution and use in source and binary forms, with or without                                       *
 *  modification, are permitted provided that the following conditions are met:                              *
 *                                                                                                           *
 *      * Redistributions of source code must retain the above copyright notice,                             *
 *        this list of conditions and the following disclaimer.                                              *
 *      * Redistributions in binary form must reproduce the above copyright                                  *
 *        notice, this list of conditions and the following disclaimer in the                                *
 *        documentation and/or other materials provided with the distribution.                               *
 *      * Neither the name of Intel Corporation nor the names of its contributors                            *
 *        may be used to endorse or promote products derived from this software                              *
 *        without specific prior written permission.                                                         *
 *                                                                                                           *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"                              *
 *  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE                                *
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE                           *
 *  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE                              *
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL                               *
 *  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR                               *
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER                               *
 *  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,                            *
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE                            *
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                                     *
 *                                                                                                           *
 *************************************************************************************************************
 *                                                                                                           *
 *  Module name:                                                                                             *
 *      freebsddriveros_i.c                                                                                  *
 *                                                                                                           *
 *  Abstract:                                                                                                *
 *      This file contains freebsddriveros_i.c                                                               *
 *                                                                                                           *
 ************************************************************************************************************/

#include <naltypes.h>
#include <nalcodes.h>
#include <os_i.h>
#include <freebsdos_i.h>
#include <freebsddevice_i.h>
#include <freebsdtypes.h>
#include <sys/types.h>
#include <machine/atomic.h>
#include <machine/stdarg.h>

#include <sys/param.h>
#include <sys/module.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/uio.h>
#include <sys/types.h>
#include <sys/malloc.h>
#include <sys/param.h>
#include <sys/lock.h>
#include <sys/mutex.h>

#include <machine/bus.h>

#include <sys/bus.h>
#include <dev/pci/pcireg.h>
#include <dev/pci/pcivar.h>

#include <sys/param.h>
#include <vm/vm.h>
#include <vm/pmap.h>

extern struct mtx Global_AtomicTestSetSpinLock;
extern BOOLEAN    Global_EnabledDebugPrint;

/***************************************************************************
**
** Name:            NalEnableDebugPrint()
**
** Description:
** Arguments:
** Returns:
**
****************************************************************************/
VOID
NalEnableDebugPrint(
    IN BOOLEAN      Enable
    )
{
    Global_EnabledDebugPrint = Enable;
}

/***************************************************************************
**
** Name:            NalDebugPrint()
**
** Description:
** Arguments:
** Returns:
**
****************************************************************************/
NAL_STATUS
NalDebugPrint(
    IN CHAR* Format,
    ...
    )
{
    NAL_STATUS NalStatus = NAL_NOT_ENABLED;
    va_list    Arguments;

    if(Global_EnabledDebugPrint == TRUE)
    {
        va_start(Arguments,Format);
        vprintf(Format,Arguments);
        va_end(Arguments);
        NalStatus = NAL_SUCCESS;
    }

    return NalStatus;
}

/***************************************************************************
**
** Name:            NalDebugPrintSupport()
**
** Description:
** Arguments:
** Returns:
**
****************************************************************************/
NAL_STATUS
NalDebugPrintSupport(
    OUT UINT32* DebugCapabilities
    )
{
    return NAL_NOT_IMPLEMENTED;
}

/***************************************************************************
**
** Name:            NalKMemSet()
**
** Description:     Sets Dest to the value of Value for Size length.
**
** Arguments:       Dest     = Pointer to the address to set
**                  Value    = Integer value to set.
**                  Size     = Bytes to set
**
** Returns:         Dest
**
****************************************************************************/
KVOID*
NalKMemset(
    IN      KVOID*      Dest,
    IN      int         Value,
    IN      UINTN       Size
    )
{
    return memset(Dest, Value, Size);
}

/***************************************************************************
**
** Name:            NalKtoUMemcpy()
**
** Description:     Copies a buffer from Kernel to User space of Size bytes.
**
** Arguments:       Source     = Pointer to the Kernel Address.
**                  Dest       = Pointer to the User Address.
**                  Size       = Bytes to Copy
**
** Returns:         Dest, or NULL if there was an error
**
****************************************************************************/
VOID*
NalKtoUMemcpy(
    IN      VOID*       Dest,
    IN      KVOID*      Source,
    IN      UINTN       Size
    )
{
    VOID * ResultPointer = NULL;
    int    ErrorCode     = 0;

    ErrorCode = copyout(Source , Dest , Size);

    if(ErrorCode == 0)
    {
        ResultPointer = Dest;
    }

    return ResultPointer;
}

/***************************************************************************
**
** Name:            NalKtoKMemcpy()
**
** Description:     Does a kernel mode memcpy between two kernel mode buffers.
**
** Arguments:       Source     = Pointer to the Kernel Address.
**                  Dest       = Pointer to the Kenerl Address.
**                  Size       = Bytes to Copy
**
** Returns:         Dest
**
****************************************************************************/
KVOID*
NalKtoKMemcpy(
    IN      KVOID*      Dest,
    IN      KVOID*      Source,
    IN      UINTN       Size
    )
{
    return memcpy(Dest, Source, Size);
}

/***************************************************************************
**
** Name:            NalUtoKMemcpy()
**
** Description:     Copies a buffer from User to Kernel space of Size bytes.
**
** Arguments:       Source     = Pointer to the User Address.
**                  Dest       = Pointer to the Kernel Address.
**                  Size       = Bytes to Copy
**
** Returns:         Dest or NULL if error
**
****************************************************************************/
KVOID*
NalUtoKMemcpy(
    IN      KVOID*      Dest,
    IN      VOID*       Source,
    IN      UINTN       Size
    )
{
    VOID * ResultPointer = NULL;
    int    ErrorCode     = 0;

    ErrorCode = copyin(Source , Dest , Size);

    if(ErrorCode == 0)
    {
        ResultPointer = Dest;
    }

    return ResultPointer;
}

/***************************************************************************
**
** Name:            NalReadPort8()
**
** Description:     Reads an 8 bit value from a specified I/O Port
**
** Arguments:       Port = the I/O port from which to read.
**
** Returns:         The data read.
**
****************************************************************************/
UINT8
NalReadPort8(
    IN  PORT_ADDR Port
    )
{
    return 0;
}

/***************************************************************************
**
** Name:            NalReadPort16()
**
** Description:     Reads a 16bit value from a specified I/O Port
**
** Arguments:       Port = the I/O port from which to read.
**
** Returns:         The data read.
**
****************************************************************************/
UINT16
NalReadPort16(
    IN  PORT_ADDR Port
    )
{
    return 0;
}

/***************************************************************************
**
** Name:            NalReadPort32()
**
** Description:     Reads a 32bit value from a specified I/O Port
**
** Arguments:       Port = the I/O port from which to read.
**
** Returns:         The data read.
**
****************************************************************************/
UINT32
NalReadPort32(
    IN  PORT_ADDR Port
    )
{
    return 0;
}

/***************************************************************************
**
** Name:            NalWritePort8()
**
** Description:     Writes an 8bit value to a specified I/O Port
**
** Arguments:       Port  = the I/O port to which to write.
**                  Value = the value to write.
**
** Returns:         TRUE if written, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
NalWritePort8(
    IN  PORT_ADDR   Port,
    IN  UINT8       Value
    )
{
    return FALSE;
}

/***************************************************************************
**
** Name:            NalWritePort16()
**
** Description:     Writes a 16bit value to a specified I/O Port
**
** Arguments:       Port  = the I/O port to which to write.
**                  Value = the value to write.
**
** Returns:         TRUE if written, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
NalWritePort16(
    IN  PORT_ADDR   Port,
    IN  UINT16      Value
    )
{
    return FALSE;
}

/***************************************************************************
**
** Name:            NalWritePort32()
**
** Description:     Writes a 32 bit value to a specified I/O Port
**
** Arguments:       Port  = the I/O port to which to write.
**                  Value = the value to write.
**
** Returns:         TRUE if written, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
NalWritePort32(
    IN  PORT_ADDR   Port,
    IN  UINT32      Value
    )
{
    return FALSE;
}

/***************************************************************************
**
** Name:            NalGetTimeStamp()
**
** Description:     Returns the TSC (Time Stamp Counter - IA32) or the ITC
**                  (Interval Time Counter - IA64).
**
** Arguments:       NONE
**
** Returns:         UINT64 - 64-bit time stamp that represents the number of
**                      clock increments since the CPU has started.
**
****************************************************************************/
UINT64
NalGetTimeStamp(VOID)
{
    return 0;
}

/***************************************************************************
**
** Name:            NalGetTimeStampsPerMicrosecond()
**
** Description:     Returns the number of time stamps in each microsecond.
**
** Arguments:       NONE
**
** Returns:         UINT64 - The number of time stamps that will occur during
**                           a microsecond.
**
** Note:            This is also implemented in user-mode of the FReeBSD OS Interface.
**                  It is only included here because the NAL IOCTL interface requires it.
****************************************************************************/
UINT64
NalGetTimeStampsPerMicrosecond(VOID)
{
    return (UINT64)0;
}
/***************************************************************************
**
** Name:            _NalAllocateMemory()
**
** Description:     Allocates the specified number of pagable bytes.
**
** Arguments:       ByteCount  = Number of bytes to allocate.
**                  NamedLocator = Name such as File name, __FILE__ works.
**                  LineNumber = Line Number in the code file. __LINE__ works.
**
** Returns:         Pointer to the memory buffer, NULL if failed.
**
** Note:            This is used ONLY to allocate kernel memory from functions within
**                  the kernel portion of the NAL.  No IOCTL call will resolve to this.
**                  Instead, they will use malloc.
****************************************************************************/
VOID*
_NalAllocateMemory(
    IN  UINT32      ByteCount,
    IN  CHAR*       NamedLocator,
    IN  UINT32      LineNumber
    )
{
    return 0;
}

/***************************************************************************
**
** Name:            _NalFreeMemory()
**
** Description:     Frees the memory allocated at the buffer address.
**
** Arguments:       Address  = Pointer to the address to free.
**                  NamedLocator = Name such as File name, __FILE__ works.
**                  LineNumber = Line Number in the code file. __LINE__ works.
**
** Returns:         Nothing.
**
** Note:            This is used ONLY to free kernel memory allocated from within
**                  the kernel portion of the NAL.  No IOCTL call will resolve to this.
****************************************************************************/
VOID
_NalFreeMemory(
    IN  VOID*       Address,
    IN  CHAR*       NamedLocator,
    IN  UINT32      LineNumber
    )
{

}

/***************************************************************************
**
** Name:            NalDelayMilliseconds()
**
** Description:     Stops the thread for a specified amt of milliseconds
**
** Arguments:       Milliseconds = Number of milliseconds to pause.
**
** Returns:         Nothing.
**
****************************************************************************/
VOID
NalDelayMilliseconds(
    IN  UINT32 Milliseconds
    )
{
}

/***************************************************************************
**
** Name:            NalDelayMicroseconds()
**
** Description:     Stops the thread for a specified amt of microseconds
**
** Arguments:       Microseconds = Number of microseconds to pause.
**
** Returns:         Nothing.
**
****************************************************************************/
VOID
NalDelayMicroseconds(
    IN  UINT32 Microseconds
    )
{

}

/***************************************************************************
**
** Name:            NalGetPhysicalMemoryAddress()
**
** Description:     Returns the Physical address associated with this virtual
**                  address. This address can be used for DMA by HW.
**
** Arguments:       Address  = Pointer to the address to resolve.
**
** Returns:         Physical Address.
**
** Note:            EFI maps physical addresses to their virtual addresses,
**                  so this returns the virtual address casted.
**
****************************************************************************/
NAL_PHYSICAL_ADDRESS
NalGetPhysicalMemoryAddress(
    IN  KVOID* VirtualAddress
    )
{
    return 0;
}

/***************************************************************************
**
** Name:            NalAtomicIncrement32()
**
** Description:     Atomically increments a UINT32 Pointer.
**
** Arguments:       Address  = Pointer to the address to increment
**
** Returns:         Value at incremented Address
**
** Note:            This assumes that the Address value is a user-mode value. This function cannot
**                  be called from within this driver to work on a kernel address. It's using
**                  copy_to_user/copy_from_user
****************************************************************************/
UINT32
NalAtomicIncrement32(
    IN  UINT32* Address
    )
{
    atomic_add_32(Address , 1);
    return *Address;
}

/***************************************************************************
**
** Name:            NalAtomicDecrement32()
**
** Description:     Atomically decrements a UINT32 Pointer.
**
** Arguments:       Address  = Pointer to the address to decrement
**
** Returns:         Value at decremented Address
**
** Note:            This assumes that the Address value is a user-mode value. This function cannot
**                  be called from within this driver to work on a kernel address. It's using
**                  copy_to_user/copy_from_user
****************************************************************************/
UINT32
NalAtomicDecrement32(
    IN  UINT32* Address
    )
{
    atomic_subtract_32(Address,1);
    return *Address;
}

/***************************************************************************
**
** Name:            NalAtomicTestSet32()
**
** Description:     If "Address" contains the same value as "Test", this will
**                  set "Address" to contain "Set" instead.
**
** Arguments:       Address  = Pointer to the address to test & set
**                  Test     = 32bit value to test
**                  Set      = 32bit value to set
**
** Returns:         Value at prior to being modified Address
**
** Note:            This assumes that the Address value is a user-mode value. This function cannot
**                  be called from within this driver to work on a kernel address. It's using
**                  copy_to_user/copy_from_user
**
****************************************************************************/
UINT32
NalAtomicTestSet32(
    IN  UINT32* Address,
    IN  UINT32  Test,
    IN  UINT32  Set
    )
{
    UINT32  ReturnValue  = 0;
    UINT32  AddressLocal = 0;

    if(Address != NULL)
    {
        /* FreeBSD doesnt provide this API, so we use a SpinLock */
        mtx_lock(&Global_AtomicTestSetSpinLock);

        copyin(Address , &AddressLocal , sizeof(UINT32));

        /* Return *Address regardless if changed or not. */
        ReturnValue = AddressLocal;

        /* Test now, then set if true. */
        if(AddressLocal == Test)
        {
            AddressLocal = Set;
        }

        copyout(&AddressLocal , Address , sizeof(UINT32));

        /* Release the spinlock */
        mtx_unlock(&Global_AtomicTestSetSpinLock);
    }
    return ReturnValue;
}
